#include <windows.h>
#include <tchar.h>
#include "FindError.h"

/* Function: GetFormatLine
   OUT szValue: Format Line; see the *.h file for description of legal values
   Return: TRUE - success; FALSE - error
*/
BOOL WINAPI GetFormatLine(TCHAR* szValue)
{
	/* This is the dummy line just for testing */
	_tcscpy(szValue, "$FILE$($LINE$) : $ERROR$: $MSG$");
	return TRUE;
}

/* Function: OnFindError
   IN hWnd: Window handle of the rich edit
   OUT pDebugInfo: Structure contains file name, line, error code, and message
   Return: TRUE - success; FALSE - error
*/

BOOL WINAPI OnFindError(HWND hWnd, LDEBUG_INFO pDebugInfo)
{
	TCHAR lpszLine[MAX_STRING]; 
	TCHAR lpszFormat[MAX_STRING];
	DEBUG_INFO DebugInfo;
	LTOKEN_LIST pTokenList = NULL;

	/* Get the format line */
	if(!GetFormatLine(lpszFormat))
	{
//		MessageBeep();
		return FALSE;
	}

	/* Parse the format line */
	if(!ParseFormatLine(lpszFormat, &pTokenList))
	{
//		MessageBeep();
		DeleteTokenList(&pTokenList);
		return FALSE;
	}

	/* Get the line from the rich edit */
	if(!RetrieveLine(hWnd, lpszLine))
	{
//		MessageBeep();
		return FALSE;
	}


	/* Parse the line from the rich edit */
	if(!ParseLine(lpszLine, &pTokenList))
	{
//		MessageBeep();
		DeleteTokenList(&pTokenList);
		return FALSE;
	}

	/* fill in the debug info to display */
	if(!FillInfo(&DebugInfo, &pTokenList))
	{
//		MessageBeep();
		DeleteTokenList(&pTokenList);
		return FALSE;
	}
	
	/* pass it to the pointer */
	*pDebugInfo = DebugInfo;
	/* Don't forget to delete the Token list */
	DeleteTokenList(&pTokenList);
	return TRUE;
}

/* Function: RetrieveLine
   IN hWnd: Window handle of the rich edit
   OUT lpszLine: line from the rich edit 
   Return: TRUE - success; FALSE - error
*/

BOOL WINAPI RetrieveLine(HWND hWnd, LPSTR lpszLine)
{
	LONG nStartChar, nEndChar;
	CHARRANGE CharRange;

	/* Get the index of the first char on the line */
	nStartChar = ::SendMessage(hWnd, EM_LINEINDEX, (WPARAM) -1, (LPARAM) 0);
	if(nStartChar == -1)
		return FALSE;
	/* get the length of the line and find the index of the last char */
	nEndChar = ::SendMessage(hWnd, EM_LINELENGTH, (WPARAM) -1, (LPARAM) 0);
	nEndChar += nStartChar;

	/* hide selection so that we can verify the format of the line */
	::SendMessage(hWnd, EM_HIDESELECTION, (WPARAM) TRUE, (LPARAM) FALSE);
	/* setup the range of characters to select */
	CharRange.cpMin = nStartChar;
	CharRange.cpMax = nEndChar;

	/* select the range */
	::SendMessage(hWnd, EM_EXSETSEL, (WPARAM) 0, (LPARAM) (CHARRANGE FAR*) &CharRange);

	/* get the selected text */
	::SendMessage(hWnd, EM_GETSELTEXT, (WPARAM) 0, (LPARAM) (LPSTR) lpszLine);
	
	return TRUE;
}

/* Function: ParseLine
   IN lpszLine: line from the rich edit
   IN/OUT pTokenList: token list 
   Return: TRUE - success; FALSE - error
*/
BOOL WINAPI ParseLine(LPCTSTR lpszLine, LTOKEN_LIST* pTokenList)
{
	LTOKEN_LIST pNode = NULL; 	/* Node points to the separator tokens */ 
	LTOKEN_LIST pPrev = NULL; 	/* Prev points to the prev valid token */
	TCHAR* cp;
	TCHAR* szValue;
	int nIndex = -1;
	TCHAR szLine[256];

	/* check if the line is valid */
	if(lpszLine[0] == '\0')
	{
		return FALSE;
	}

	/* check if the token list is valid */
	if(*pTokenList == NULL)
		return FALSE;

	/* copy the line to a local variable */
	_tcscpy(szLine, lpszLine);
	/* add the terminating token to the line */
	_tcscat(szLine, kTokenStringTerm);

	/* assign both pointers to the token list */
	pNode = *pTokenList;
	pPrev = *pTokenList;

	/* assign the char pointer to the line */
	cp = szLine;

	/* go through the token list */

	while(pNode != NULL)
	{
		switch(pNode->Token)
		{
			case SEPARATOR_TOKEN:
				/* if token is a separator
				     find the index of the separator string
					 copy the line upto the separator string
					 cut it off
					 if the previous token is not a separator
					  copy the cut off line into its value
					 cut off the separator
					 move both pointer up one
					 */
				nIndex = Find(cp, pNode->szValue);
				if(nIndex == -1)
					return FALSE;
				szValue = cp;	
				cp += nIndex * sizeof(TCHAR);
				*cp = NULL;
				if(pNode == pPrev->pNext)
					_tcscpy(pPrev->szValue, szValue);
				cp += _tcslen(pNode->szValue) - 1;
				*cp = NULL;
				cp++;
				pNode = pNode->pNext;
				pPrev = pNode;
			break;
			default:
				/* anything other than the separator
				 move the current pointer one up
				 and place the previous pointer in its place */
				pNode = pNode->pNext;
				while(pPrev->pNext != pNode && pPrev->pNext != NULL)
				pPrev = pPrev->pNext;
			break;
		}
	}
	return TRUE;
}

/* Function: AddTokenToList
   IN Token: type of token
   IN szValue: token
   IN/OUT: Token List
   Return: TRUE - success; FALSE - error
*/
BOOL WINAPI AddTokenToList(TOKEN Token, TCHAR* szValue, LTOKEN_LIST* pTokenList)
{
	LTOKEN_LIST pNode = NULL;
	LTOKEN_LIST pPtr = NULL;

	pPtr = CreateToken(Token, szValue);
	if(pPtr == NULL)
		return FALSE;

	if(*pTokenList == NULL)
	{
		*pTokenList = pPtr;
		return TRUE;
	}
	pNode = *pTokenList;
	while(pNode->pNext != NULL)
	{
		pNode = pNode->pNext;
	}

	pNode->pNext = pPtr;

	return TRUE;
}

/* Function: FillInfo
   IN/OUT pDebugInfo: Debug Structure
   IN pTokenList: Token List
   Return: TRUE - success; FALSE - error
*/
BOOL WINAPI FillInfo(LDEBUG_INFO pDebugInfo, LTOKEN_LIST* pTokenList)
{
	LTOKEN_LIST pToken = NULL;

	pToken = *pTokenList;
	while(pToken != NULL)
	{
		switch(pToken->Token)
		{
			case FILEPATH_TOKEN:
				_tcscpy(pDebugInfo->szFileName, pToken->szValue);
			break;
			
			case LINE_TOKEN:
				pDebugInfo->nLine = atol(pToken->szValue);
			break;

			case ERRORCODE_TOKEN:
				_tcscpy(pDebugInfo->szErrorCode, pToken->szValue);
			break;
			
			case ERRORMSG_TOKEN:
				_tcscpy(pDebugInfo->szErrorMsg, pToken->szValue);
			break;
		}
		pToken = pToken->pNext;
		
	}

	return TRUE;
}

/* Function: CreateToken
   IN token: token type
   IN szValue: token
   Return: pointer to a new token.  If error returns NULL
*/
LTOKEN_LIST WINAPI CreateToken(TOKEN token, TCHAR* szValue)
{
	LTOKEN_LIST pNode = NULL;

	pNode = (LTOKEN_LIST) malloc(sizeof(TOKEN_LIST));
	if(pNode != NULL)
	{
		pNode->Token = token;
		_tcscpy(pNode->szValue, szValue);
		pNode->pNext = NULL;
	}

	return pNode;
}

/* Function: DeleteTokenList
   IN/OUT pTokenList: Token List
   Return: TRUE - success; FALSE - error
*/
BOOL WINAPI DeleteTokenList(LTOKEN_LIST* pTokenList)
{
	LTOKEN_LIST pToken = NULL, pTokenNext = NULL;

	pToken = *pTokenList;
	while(pToken != NULL)
	{
		pTokenNext = pToken->pNext;
		free(pToken);
		pToken = pTokenNext;
		
	}

	*pTokenList = NULL;
	pTokenList = NULL;
	return TRUE;
}


/* Function: Find
   IN lpszString: string
   IN lpszSub: substring to find
   Return: index of substring in the string or -1
*/
int WINAPI Find(LPCTSTR lpszString, LPCTSTR lpszSub)
{

	/* find first matching substring */
	LPTSTR lpsz = _tcsstr(lpszString, lpszSub);

	/* return -1 for not found, distance from beginning otherwise */
	return (lpsz == NULL) ? -1 : (int)(lpsz - lpszString);
}

/* Function: ParseFormatLine
   IN lpszFormatLine: Format Line
   IN/OUT pTokenList: Token List
   Return: TRUE - success; FALSE - error
*/
BOOL WINAPI ParseFormatLine(LPCTSTR lpszFormatLine, LTOKEN_LIST* pTokenList)
{
	TCHAR* cp;
	BOOL bIsToken = FALSE;
	TCHAR* szValue;
	TCHAR temp;

	cp = (TCHAR*)lpszFormatLine;

	/* parse the line and using list of tokens */
	while(*cp != NULL)
	{
		if(*cp == kTokenCharTerm)
		{
			if(bIsToken)
				bIsToken = FALSE;
			else
				bIsToken = TRUE;
		}
		else
			bIsToken = FALSE;
		
		szValue = cp;
		cp++;
		while(*cp != kTokenCharTerm && *cp != NULL)
			cp++;
		if(bIsToken)
		{
			cp++;
			temp = *cp;
			*cp = '\0';
			if(!_tcscmp(szValue, kFileToken))
			{
				AddTokenToList(FILEPATH_TOKEN, szValue, pTokenList);
			}
			else if(!_tcscmp(szValue, kErrorToken))
			{
				AddTokenToList(ERRORCODE_TOKEN, szValue, pTokenList);
			}
			else if(!_tcscmp(szValue, kLineToken))
			{
				AddTokenToList(LINE_TOKEN, szValue, pTokenList);
			}
			else if(!_tcscmp(szValue, kMsgToken))
			{
				AddTokenToList(ERRORMSG_TOKEN, szValue, pTokenList);
			}
			else
			{
				return FALSE;
			}
		}
		else
		{
			/* AddTokenToList as separator */
			temp = *cp;
			*cp = '\0';
			AddTokenToList(SEPARATOR_TOKEN, szValue, pTokenList);
		}
		szValue = '\0';
		*cp = temp;
	}

	AddTokenToList(SEPARATOR_TOKEN, kTokenStringTerm, pTokenList);

	return TRUE;
}